<?php
defined('BASEPATH') OR exit('No direct script access allowed');

class Teams_model extends CI_Model {
    public function get_settings() {
        return $this->db->get('teams_settings')->row();
    }

    public function save_settings($data) {
        if ($this->db->get('teams_settings')->num_rows() > 0) {
            $this->db->update('teams_settings', $data);
        } else {
            $this->db->insert('teams_settings', $data);
        }
    }

    public function save_tokens($token_data) {
        $data = [
            'access_token' => $token_data['access_token'],
            'refresh_token' => $token_data['refresh_token'],
            'token_expiry' => date('Y-m-d H:i:s', time() + $token_data['expires_in'])
        ];
        $this->db->update('teams_settings', $data);
    }

    public function get_access_token()
    {
        $settings = $this->db->get('teams_settings')->row();

        if (!$settings) {
            throw new Exception("Microsoft Teams token not found in settings.");
        }

        $current_time = time();
        $token_expiry = strtotime($settings->token_expiry);

        if ($current_time >= $token_expiry - 300 || empty($settings->access_token)) {
            if (empty($settings->refresh_token)) {
                throw new Exception("No refresh token available. Please reconnect to Microsoft Teams.");
            }

            $refresh_token = $settings->refresh_token;

            try {
                $client = new \GuzzleHttp\Client(['timeout' => 30]);
                
                $tenant = (!empty($settings->tenant_id)) ? $settings->tenant_id : 'common';
                
                $response = $client->post("https://login.microsoftonline.com/{$tenant}/oauth2/v2.0/token", [
                    'form_params' => [
                        'grant_type'    => 'refresh_token',
                        'refresh_token' => $refresh_token,
                        'client_id'     => $settings->client_id,
                        'client_secret' => $settings->client_secret,
                        'scope'         => 'offline_access OnlineMeetings.ReadWrite Calendars.ReadWrite User.Read'
                    ],
                    'http_errors' => false
                ]);

                $status = $response->getStatusCode();
                $token_data = json_decode($response->getBody(), true);

                if ($status !== 200 || !isset($token_data['access_token'])) {
                    error_log("Failed to refresh access token: " . $response->getBody());
                    throw new Exception("Failed to refresh access token. Response: " . substr($response->getBody(), 0, 200));
                }

                $this->save_tokens($token_data);

                return $token_data['access_token'];
            } catch (Exception $e) {
                throw new Exception("Failed to refresh token: " . $e->getMessage());
            }
        }

        return $settings->access_token;
    }

    public function save_meeting($meeting_data) {
        try {
            $existing = $this->db->get_where('teams_meetings', ['meeting_id' => $meeting_data['id']])->row();
            if ($existing) {
                $data = [
                    'title' => $meeting_data['subject'],
                    'description' => isset($meeting_data['body']['content']) ? $meeting_data['body']['content'] : null,
                    'start_time' => isset($meeting_data['startDateTime']) ? $meeting_data['startDateTime'] : $meeting_data['start']['dateTime'],
                    'end_time' => isset($meeting_data['endDateTime']) ? $meeting_data['endDateTime'] : $meeting_data['end']['dateTime'],
                    'join_url' => isset($meeting_data['joinUrl']) ? $meeting_data['joinUrl'] : 
                                (isset($meeting_data['onlineMeeting']['joinUrl']) ? $meeting_data['onlineMeeting']['joinUrl'] : ''),
                ];
                
                $this->db->where('meeting_id', $meeting_data['id']);
                $this->db->update('teams_meetings', $data);
                return true;
            }
            
            $data = [
                'meeting_id' => $meeting_data['id'],
                'title' => $meeting_data['subject'],
                'description' => isset($meeting_data['body']['content']) ? $meeting_data['body']['content'] : null,
                'start_time' => isset($meeting_data['startDateTime']) ? $meeting_data['startDateTime'] : $meeting_data['start']['dateTime'],
                'end_time' => isset($meeting_data['endDateTime']) ? $meeting_data['endDateTime'] : $meeting_data['end']['dateTime'],
                'join_url' => isset($meeting_data['joinUrl']) ? $meeting_data['joinUrl'] : 
                            (isset($meeting_data['onlineMeeting']['joinUrl']) ? $meeting_data['onlineMeeting']['joinUrl'] : ''),
                'created_by' => user()->id,
                'created_at' => date('Y-m-d H:i:s')
            ];
            
            $this->db->insert('teams_meetings', $data);
            return true;
        } catch (Exception $e) {
            return false;
        }
    }

    // Get Meeting Details and Recording URL
    public function get_meeting_recording($meeting_id) {
        $access_token = $this->get_access_token();
        if (!$access_token) {
            return null;
        }

        $client = new GuzzleHttp\Client();
        try {
            $response = $client->get("https://graph.microsoft.com/v1.0/me/onlineMeetings/$meeting_id", [
                'headers' => [
                    'Authorization' => 'Bearer ' . $access_token,
                    'Content-Type' => 'application/json'
                ]
            ]);

            $meeting = json_decode($response->getBody(), true);
            return $meeting['recordings'] ?? null;
        } catch (Exception $e) {
            log_message('error', 'Error fetching meeting recording: ' . $e->getMessage());
            return null;
        }
    }

    // Fetch Recording File from OneDrive
    public function get_recording_file($recording_id) {
        $access_token = $this->get_access_token();
        if (!$access_token) {
            return null;
        }

        $client = new GuzzleHttp\Client();
        try {
            $response = $client->get("https://graph.microsoft.com/v1.0/me/drive/items/$recording_id", [
                'headers' => [
                    'Authorization' => 'Bearer ' . $access_token,
                    'Content-Type' => 'application/json'
                ]
            ]);

            return json_decode($response->getBody(), true);
        } catch (Exception $e) {
            log_message('error', 'Error fetching recording file: ' . $e->getMessage());
            return null;
        }
    }
}